/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/io.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier PCM Driver");
MODULE_LICENSE("GPL");

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 35)
/**
 * ʤؿǤprintk ̵뤿˻Ȥޤ
 */
int no_printk(const char *fmt, ...)
{
	return 0;
}
#endif

/**
 * ꤷ쥸ΰ֤˥ޥåפޤ
 * 
 * @param mr    ޥåԥ󥰾
 * @param size  ޥåפΰΥ
 * @param paddr ޥåפΰʪɥ쥹
 * @return  0ԤΥ顼
 */
int pcm_mmap(struct mem_mapping *mr, size_t size, unsigned long paddr)
{
	void *mapping = NULL;
	int result = 0;
	
	mapping = ioremap_nocache(paddr, size);
	if (mapping == NULL) {
		PRINTF_WARN("failed to map pcm registers, error %d\n", 
			result);
		result = -EINVAL;
		goto err_out1;
	}
	
	mr->size = size;
	mr->paddr = paddr;
	mr->mapping = mapping;
	
	return 0;
	
err_out1:
	return result;
}

/**
 * ֤˥ޥåפ줿쥸ΰΥޥåפޤ
 * 
 * @param mr ޥåԥ󥰾
 * @return  0ԤΥ顼
 */
int pcm_munmap(struct mem_mapping *mr)
{
	int result = 0;
	
	if (mr->mapping == NULL) {
		PRINTF_WARN("already unmapped pcm registers.\n");
		result = -EINVAL;
		goto err_out1;
	}
	
	iounmap(mr->mapping);
	mr->size = 0;
	mr->paddr = 0;
	mr->mapping = NULL;
	
	return 0;

err_out1:
	return result;
}

/**
 * ʪɥ쥹 CPU 饢ǽʥɥ쥹Ѵ롣
 * 
 * @param mr    ޥåԥ󥰾
 * @param paddr ʪɥ쥹
 * @return CPU ۥɥ쥹
 */
void *regs_phys_to_virt(const struct mem_mapping *mr, unsigned long paddr)
{
	if (paddr < mr->paddr || (mr->paddr + mr->size) < paddr) {
		PRINTF_WARN("%s: 0x%08lx is out of bounds, return 0.\n", 
			__func__, paddr);
		return 0;
	}
	
	return (void *)((u8 *)mr->mapping + (paddr - mr->paddr));
}

u32 pcm_readl(const struct mem_mapping *mr, unsigned long paddr)
{
	volatile u32 *ptr;
	
	if (paddr < mr->paddr || (mr->paddr + mr->size) < paddr) {
		PRINTF_WARN("%s: 0x%08lx is out of bounds, return 0.\n", 
			__func__, paddr);
		return 0;
	}
	
	ptr = (void *)((u8 *)mr->mapping + (paddr - mr->paddr));
	
	return *ptr;
}

void pcm_writel(const struct mem_mapping *mr, u32 val, unsigned long paddr)
{
	volatile u32 *ptr;
	
	if (paddr < mr->paddr || (mr->paddr + mr->size) < paddr) {
		PRINTF_WARN("%s: 0x%08lx is out of bounds, ignored.\n", 
			__func__, paddr);
		return;
	}
	
	ptr = (void *)((u8 *)mr->mapping + (paddr - mr->paddr));
	
	*ptr = val;
}

/**
 * ХȰ֤žޤ
 * 
 * v      : 0x00010203
 * return : 0x03020100
 * 
 * @param v Ǥդ
 * @return ХȰ֤ž
 */
u32 rev32(u32 v)
{
        v = ((v & 0x00ff00ff) <<  8) | ((v & 0xff00ff00) >>  8);
        v = ((v & 0x0000ffff) << 16) | ((v & 0xffff0000) >> 16);

        return v;
}

/**
 * 4ХȤĥХȰ֤žʤ饳ԡޤ
 * 
 *   src : 00 01 02 03  04 05 06 07  ...
 *   dest: 03 02 01 00  07 06 05 04  ...
 * 
 * @param dest ԡХåե
 * @param src  ԡХåե
 * @param n    ԡ륵ʥХñ̡4ܿꤷƤ
 * @return ԡΥХåե
 */
void *memcpy_rev(void *dest, const void *src, size_t n)
{
	u32 *d;
	const uint32_t *s;
	size_t i;

	if ((n & 3) != 0) {
		PRINTF_WARN("maybe overrun...(n:%d)\n", n);
	}

	d = dest;
	s = src;
	for (i = 0; i < n / 4; i++) {
		d[i] = rev32(s[i]);
	}

	return dest;
}

/**
 * 󥰥Хåեޤ
 * 󥰥ХåեݻǤΥǡϡ
 * ХåեĹ - 1ʤĤޤ len - 1ˤȤʤޤ
 * 
 * @param r   󥰥Хåե
 * @param buf 󥰥ХåեȤƴХåե
 * @param len ХåեĹ
 * @return ʤ 0顼ξ -1
 */
int init_ringbuf(struct ringbuf *r, unsigned char *buf, size_t len)
{
	if (r == NULL) {
		PRINTF_WARN("cannot init the ring buffer(NULL).\n");
		return -1;
	}
	
	r->start = buf;
	r->len = len;
	r->rd = 0;
	r->wr = 0;
	
	return 0;
}

/**
 * 󥰥Хåեθߤɤ߼֤ޤ
 * 
 * @param r 󥰥Хåե
 * @return 0 ʾθߤɤ߼֡顼ξ -1
 */
loff_t get_rp_ringbuf(const struct ringbuf *r)
{
	if (r == NULL) {
		PRINTF_WARN("cannot get the read pos of ring buffer(NULL).\n");
		return -1;
	}
	
	return r->rd;
}

/**
 * 󥰥Хåեɤ߼֤ꤷޤ
 * 
 * @param r   󥰥Хåե
 * @param pos ɤ߼
 * @return ʤ 0顼ξ -1
 */
int set_rp_ringbuf(struct ringbuf *r, loff_t pos)
{
	if (r == NULL) {
		PRINTF_WARN("cannot set the read pos of ring buffer(NULL).\n");
		return -1;
	}
	
	r->rd = pos % r->len;
	
	return 0;
}

/**
 * 󥰥Хåեθߤν񤭹֤߰ޤ
 * 
 * @param r 󥰥Хåե
 * @return 0 ʾθߤν񤭹߰֡顼ξ -1
 */
loff_t get_wp_ringbuf(const struct ringbuf *r)
{
	if (r == NULL) {
		PRINTF_WARN("cannot get the write pos of ring buffer(NULL).\n");
		return -1;
	}
	
	return r->wr;
}

/**
 * 󥰥Хåեν񤭹֤߰ꤷޤ
 * 
 * @param r   󥰥Хåե
 * @param pos 񤭹߰
 * @return ʤ 0顼ξ -1
 */
int set_wp_ringbuf(struct ringbuf *r, loff_t pos)
{
	if (r == NULL) {
		PRINTF_WARN("cannot set the write pos of ring buffer(NULL).\n");
		return -1;
	}
	
	r->wr = pos % r->len;
	
	return 0;
}

/**
 * 󥰥Хåե˳ǼƤǡΥХȿޤ
 * 
 * @param r 󥰥Хåե
 * @return 0 ʾγǼƤǡΥХȿ顼ξ -1
 */
size_t get_remain_ringbuf(const struct ringbuf *r)
{
	if (r == NULL) {
		PRINTF_WARN("cannot get the remains of ring buffer(NULL).\n");
		return -1;
	}
	
	return get_remain(r->rd, r->wr, r->len);
}

/**
 * 󥰥ХåեζΰΥХȿޤ
 * 
 * @param r 󥰥Хåե
 * @return 0 ʾζΰΥХȿ顼ξ -1
 */
size_t get_space_ringbuf(const struct ringbuf *r)
{
	if (r == NULL) {
		PRINTF_WARN("cannot get the space of ring buffer(NULL).\n");
		return -1;
	}
	
	return get_space(r->rd, r->wr, r->len);
}

/**
 * 󥰥Хåեꤵ줿ǡޤ
 * 
 * @param r  󥰥Хåե
 * @param d  񤭹Х
 * @return ξ 0顼ξ -1
 */
int fill_ringbuf(struct ringbuf *r, unsigned char d)
{
	if (r == NULL) {
		PRINTF_WARN("cannot fill the ring buffer(NULL).\n");
		return -1;
	}
	
	memset(r->start, d, r->len);
	
	return 0;
}

/**
 * 󥰥Хåեǡɤ߹ߤޤ
 * 
 * @param r     󥰥Хåե
 * @param buf   ɤ߹ǡǼХåե
 * @param count ɤ߹Хȿ
 * @return 0 ʾɤ߹Хȿ顼ξ -1
 */
ssize_t read_ringbuf(struct ringbuf *r, void *buf, size_t count)
{
	unsigned char *cbuf = buf;
	size_t tran;
	ssize_t result;
	
	if (r == NULL || buf == NULL) {
		return -1;
	}
	
	result = 0;
	while (result < count) {
		if (get_remain(r->rd, r->wr, r->len) == 0) {
			//under
			break;
		}
		
		tran = count - result;
		tran = min(tran, 
			get_remain_continuous(r->rd, r->wr, r->len));
		
		memcpy_rev(&cbuf[result], &r->start[r->rd], tran);
		
		result += tran;
		r->rd += tran;
		if (r->rd >= r->len) {
			r->rd = 0;
		}
	}
	
	return result;
}

/**
 * 󥰥Хåե˥ǡ񤭹ߤޤ
 * 
 * @param r     󥰥Хåե
 * @param buf   񤭹ǡǼƤХåե
 * @param count 񤭹Хȿ
 * @return 0 ʾʤн񤭹Хȿ-1 ʤХ顼
 */
ssize_t write_ringbuf(struct ringbuf *r, const void *buf, size_t count)
{
	const unsigned char *cbuf = buf;
	size_t tran;
	ssize_t result;
	
	if (r == NULL || buf == NULL) {
		return -1;
	}
	
	result = 0;
	while (result < count) {
		if (get_space(r->rd, r->wr, r->len) == 0) {
			//over
			break;
		}
		
		tran = count - result;
		tran = min(tran, 
			get_space_continuous(r->rd, r->wr, r->len));
		
		memcpy_rev(&r->start[r->wr], &cbuf[result], tran);
		
		result += tran;
		r->wr += tran;
		if (r->wr >= r->len) {
			r->wr = 0;
		}
	}
	
	return result;
}

/**
 * 󥰥Хåե 0 ǡ񤭹ߤޤ
 * 
 * @param r     󥰥Хåե
 * @param count 񤭹Хȿ
 * @return 0 ʾʤн񤭹Хȿ-1 ʤХ顼
 */
ssize_t write_silent_ringbuf(struct ringbuf *r, size_t count)
{
	size_t tran;
	ssize_t result;
	
	if (r == NULL) {
		return -1;
	}
	
	result = 0;
	while (result < count) {
		if (get_space(r->rd, r->wr, r->len) == 0) {
			//over
			break;
		}
		
		tran = count - result;
		tran = min(tran, 
			get_space_continuous(r->rd, r->wr, r->len));
		
		memset(&r->start[r->wr], 0, tran);
		
		result += tran;
		r->wr += tran;
		if (r->wr >= r->len) {
			r->wr = 0;
		}
	}
	
	return result;
}

/**
 * 󥰥Хåե顢̤Υ󥰥Хåե˥ǡ򥳥ԡޤ
 * 
 * @param dest  񤭹Υ󥰥Хåե
 * @param src   񤭹ǡǼƤ󥰥Хåե
 * @param count 񤭹Хȿ
 * @return 0 ʾʤн񤭹Хȿ-1 ʤХ顼
 */
ssize_t copy_ringbuf(struct ringbuf *dest, struct ringbuf *src, size_t count)
{
	size_t tran;
	ssize_t result;
	
	if (dest == NULL || src == NULL) {
		return -1;
	}
	
	result = 0;
	while (result < count) {
		if (get_space(dest->rd, dest->wr, dest->len) == 0) {
			//over
			break;
		}
		if (get_remain(src->rd, src->wr, src->len) == 0) {
			//under
			break;
		}
		
		tran = count - result;
		tran = min(tran, 
			get_space_continuous(dest->rd, dest->wr, dest->len));
		tran = min(tran, 
			get_remain_continuous(src->rd, src->wr, src->len));
		
		memcpy_rev(&dest->start[dest->wr], &src->start[src->rd], tran);
		
		result += tran;
		dest->wr += tran;
		if (dest->wr >= dest->len) {
			dest->wr = 0;
		}
		
		src->rd += tran;
		if (src->rd >= src->len) {
			src->rd = 0;
		}
	}
	
	return result;
}

/**
 * Get the remain bytes of buffer.
 * 
 * @param rd  read pointer of buffer
 * @param wr  write pointer of buffer
 * @param len length of buffer
 * @return the remain bytes of buffer.
 */
size_t get_remain(loff_t rd, loff_t wr, size_t len)
{
	size_t rest;
	
	if (rd <= wr) {
		rest = wr - rd;
	} else {
		rest = len - (rd - wr);
	}
	
	return rest;
}

/**
 * Get the continuous remain bytes of buffer.
 * 
 * @param rd  read pointer of buffer
 * @param wr  write pointer of buffer
 * @param len length of buffer
 * @return the remain bytes of buffer.
 */
size_t get_remain_continuous(loff_t rd, loff_t wr, size_t len)
{
	size_t rest;
	
	if (rd <= wr) {
		rest = wr - rd;
	} else {
		rest = len - rd;
	}
	
	return rest;
}

/**
 * Get the space bytes of buffer.
 * 
 * @param rd  read pointer of buffer
 * @param wr  write pointer of buffer
 * @param len length of buffer
 * @return the space bytes of buffer.
 */
size_t get_space(loff_t rd, loff_t wr, size_t len)
{
	size_t rest;
	
	if (rd <= wr) {
		rest = len - (wr - rd) - 1;
	} else {
		rest = rd - wr - 1;
	}
	
	return rest;
}

/**
 * Get the continuous space bytes of buffer.
 * 
 * @param rd  read pointer of buffer
 * @param wr  write pointer of buffer
 * @param len length of buffer
 * @return the continuous space bytes of buffer.
 */
size_t get_space_continuous(loff_t rd, loff_t wr, size_t len)
{
	size_t rest;
	
	if (rd > wr) {
		rest = rd - wr - 1;
	} else if (rd > 0) {
		rest = len - wr;
	} else {
		rest = len - wr - 1;
	}
	
	return rest;
}

/**
 * a λͤ b λͤû
 * η̤ res ˤ껲Ȥ timeval ¤Τ˳Ǽ롣
 * 
 * @param a   Ȥʤ
 * @param b   a ˲û
 * @param res ûλ
 */
void timeradd(struct timeval *a, struct timeval *b, struct timeval *res)
{
	res->tv_sec = a->tv_sec + b->tv_sec;
	res->tv_usec = a->tv_usec + b->tv_usec;
	
	if (res->tv_usec >= 1000000) {
		res->tv_sec += 1;
		res->tv_usec -= 1000000;
	}
}

/**
 * a λͤ b λͤ򸺻
 * η̤ res ˤ껲Ȥ timeval ¤Τ˳Ǽ롣
 * 
 * @param a   Ȥʤ
 * @param b   a 鸺
 * @param res λ
 */
void timersub(struct timeval *a, struct timeval *b, struct timeval *res)
{
	res->tv_sec = a->tv_sec - b->tv_sec;
	res->tv_usec = a->tv_usec - b->tv_usec;
	
	if (res->tv_usec < 0) {
		res->tv_sec -= 1;
		res->tv_usec += 1000000;
	}
}

/**
 * tvp ˤ껲Ȥ timeval ¤Τ 0 롣
 * 0 줿 timeval ¤Τϡ
 * ﵪ (Epoch; 1970-01-01 00:00:00 +0000 (UTC)) ɽ
 * 
 * @param 0 
 */
void timerclear(struct timeval *tvp)
{
	tvp->tv_sec = 0;
	tvp->tv_usec = 0;
}
